library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_signed.all;
 
library unisim ;
use unisim.vcomponents.all ;
 
entity vhdl_upl_rx8 is
  generic (
    DATA_WIDTH : integer := 8;
    DATA_DEPTH : integer := 12
  );
  port (
    pReset_n : in std_logic;
    pUPLGlobalClk : in std_logic;
    
    -- UPLからの入力を受ける
    pI0Data    : in std_logic_vector(DATA_WIDTH-1 downto 0 );
    pI0Request : in std_logic;
    pI0Ack     : out std_logic;
    pI0En      : in std_logic;
    
    -- JavaとのI/F
    pDataValid   : out std_logic;
    pDataRead    : in  std_logic;
    pMemReadData : out std_logic_vector(DATA_WIDTH-1 downto 0);
    pMemReadAddr : in  std_logic_vector(DATA_DEPTH-1 downto 0);
    pDataLength  : out std_logic_vector(DATA_DEPTH-1 downto 0);

    pDebugA : out std_logic_vector( 7 downto 0 )
    );
end vhdl_upl_rx8;

architecture RTL of vhdl_upl_rx8 is

  component simpledualportram
    generic (
      DEPTH : integer := 12;
      WIDTH : integer := 32);
    port (
      clk   : in  std_logic;
      we    : in  std_logic_vector(0 downto 0);
      raddr : in  std_logic_vector(DEPTH-1 downto 0);
      rdata : out std_logic_vector(WIDTH-1 downto 0);
      waddr : in  std_logic_vector(DEPTH-1 downto 0);
      wdata : in  std_logic_vector(WIDTH-1 downto 0)
      );
  end component;

  type UPLRxState is (IDLE, RECV, READY, TRANS);
  signal state : UPLRxState := IDLE;
  signal rx_counter : std_logic_vector(DATA_DEPTH-1 downto 0) := (others => '0');
  signal recv_length : std_logic_vector(DATA_DEPTH-1 downto 0) := (others => '0');

  signal cWe       : std_logic_vector(0 downto 0) := (others => '0');
  signal cWAddr    : std_logic_vector(DATA_DEPTH-1 downto 0) := (others => '0');
  signal next_addr : std_logic_vector(DATA_DEPTH-1 downto 0) := (others => '0');
  signal cWData    : std_logic_vector(DATA_WIDTH-1 downto 0) := (others => '0');

begin  -- RTL

  M: simpledualportram
    generic map( DEPTH => DATA_DEPTH, WIDTH => DATA_WIDTH )
    port map(
      clk   => pUPLGlobalClk,
      we    => cWe,
      raddr => pMemReadAddr,
      rdata => pMemReadData,
      waddr => cWAddr,
      wdata => cWData
      );

  pDataLength <= recv_length;
  
  process (pUPLGlobalClk)
  begin  -- process
    if pUPLGlobalClk'event and pUPLGlobalClk = '1' then  -- rising clock edge
      if pReset_n = '0' then  -- asynchronous reset (active low)
        state      <= IDLE;
        pI0Ack     <= '0';
        rx_counter <= (others => '0');
        pDataValid <= '0';
        cWAddr <= (others => '0');
        cWe <= "0";
        recv_length <= (others => '0');
      else
        case state is
          ---------------------------------------------------------------------------
          -- UPLパケットの到着を待つ
          ---------------------------------------------------------------------------
          when IDLE =>
            if pI0En = '1' then
              state  <= RECV;
              rx_counter <= rx_counter + 1;
              pI0Ack <= '0';
              cWAddr <= (others => '0');
              cWData    <= pI0Data;
              cWe    <= "1";
              next_addr <= next_addr + 1;
              recv_length <= recv_length + 1;
            else
              recv_length <= (others => '0');
              pI0Ack <= '1';
              rx_counter <= (others => '0');
              cWAddr <= (others => '0');
              cWe    <= "0";
              next_addr <= (others => '0');
            end if;
            pDataValid <= '0';
          ---------------------------------------------------------------------------
          -- UPLパケットを受信する
          ---------------------------------------------------------------------------
          when RECV =>
            pI0Ack <= '0';
            if pI0En = '0' then
              state <= READY;
              cWe   <= "0";
              pDataValid <= '1';
            else
              recv_length<= recv_length + 1;
              rx_counter <= rx_counter + 1;
              cWAddr    <= next_addr;
              cWData    <= pI0Data;
              cWe       <= "1";
              next_addr <= next_addr + 1;
            end if;
          ---------------------------------------------------------------------------
          -- Javaから読み出されるのを待つ
          ---------------------------------------------------------------------------
          when READY =>
            if pDataRead = '1' then     -- データの読み出しの開始
              pDataValid <= '0';        -- 読み出されはじめたらデータを無効化しておく
              state <= TRANS;
            end if;
          ---------------------------------------------------------------------------
          -- Javaからデータを抜きとってもらう
          ---------------------------------------------------------------------------
          when TRANS =>
            if pDataRead = '0' then     -- データの読み出しがおわった
              pI0Ack <= '1';
              state <= IDLE;
              recv_length <= (others => '0');
              pDataValid <= '0';
            end if;
          ---------------------------------------------------------------------------
          -- ここにはこないはず
          ---------------------------------------------------------------------------
          when others =>
            state <= IDLE;
        end case;
      end if;
    end if;
  end process;

end RTL;
