library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_signed.all;

library unisim ;
use unisim.vcomponents.all ;

entity vhdl_upl_tx8 is
  generic (
    DATA_WIDTH : integer := 8;
    DATA_DEPTH : integer := 12
  );
  port (
    pReset_n : in std_logic;
    pClk     : in std_logic;

    -- 出力
    pO0Data    : out std_logic_vector( DATA_WIDTH-1 downto 0 );
    pO0Request : out std_logic;
    pO0Ack     : in  std_logic;
    pO0Enable  : out std_logic;

    -- JavaとのI/F
    pSendRequest : in  std_logic;
    pBusy        : out std_logic;
    pSendLength  : in  std_logic_vector(DATA_DEPTH-1 downto 0);
    pMemWriteEn  : in  std_logic_vector(0 downto 0);
    pMemWriteData: in  std_logic_vector(DATA_WIDTH-1 downto 0);
    pMemWriteAddr: in  std_logic_vector(DATA_DEPTH-1 downto 0);

    pDebugA : out std_logic_vector(7 downto 0)

    );
end vhdl_upl_tx8;

architecture RTL of vhdl_upl_tx8 is

  component simpledualportram
    generic (
      DEPTH : integer := 12;
      WIDTH : integer := 32);
    port (
      clk   : in  std_logic;
      we    : in  std_logic_vector(0 downto 0);
      raddr : in  std_logic_vector(DEPTH-1 downto 0);
      rdata : out std_logic_vector(WIDTH-1 downto 0);
      waddr : in  std_logic_vector(DEPTH-1 downto 0);
      wdata : in  std_logic_vector(WIDTH-1 downto 0)
      );
  end component;

  -- ステートマシンのステート定義
  type stateType is ( IDLE, SEND_REQUEST, SEND_DATA );
  signal State : stateType;

  -- パケット送信ワードカウンタ
  signal SendCount : std_logic_vector( 7 downto 0 );

  -- 送信用定数定義
  signal myIpAddr     : std_logic_vector( 31 downto 0 ) := X"0a000001";  -- 送り元IPアドレス=10.0.0.1
  signal ServerIpAddr : std_logic_vector( 31 downto 0 ) := X"0a000003";  -- 送り先IPアドレス=10.0.0.3
  signal MyPort       : std_logic_vector( 15 downto 0 ) := X"4001";  -- 送り元UDPポート番号=16385
  signal ServerPort   : std_logic_vector( 15 downto 0 ) := X"4001";  -- 送り先UDPポート番号=16385

  -- pSendRequestエッジ検出のための遅延レジスタ    
  signal send_request_reg : std_logic;

  signal cRAddr : std_logic_vector(DATA_DEPTH-1 downto 0);
  signal cRDATA : std_logic_vector(DATA_WIDTH-1 downto 0);
  signal cLength: std_logic_vector(DATA_DEPTH-1 downto 0);
  
begin

  M: simpledualportram
    generic map( DEPTH => DATA_DEPTH, WIDTH => DATA_WIDTH )
    port map(
      clk   => pClk,
      we    => pMemWriteEn,
      raddr => cRAddr,
      rdata => cRData,
      waddr => pMemWriteAddr,
      wdata => pMemWriteData
      );

  process( pClk ) 
  begin
    if( pClk'event and pClk = '1' ) then
      if( pReset_n = '0' ) then
        -- リセット処理
        pO0Enable  <= '0';
        pO0Request <= '0';
        pO0Data    <= (others => '0');
        State <= IDLE;
        SendCount <= (others => '0');
        pBusy <= '0';
        cRAddr <= (others => '0');
      else
        send_request_reg <= pSendRequest;
        
        case State is 
          when IDLE =>
            -- 送信リクエストを検出する
            if( pSendRequest = '1' ) then
              State <= SEND_REQUEST;
              pBusy <= '1';
              cLength <= pSendLength;
            else
              pBusy <= '0';
            end if;
            cRAddr <= (others => '0');
            SendCount <= (others => '0');
          when SEND_REQUEST =>
            -- UPLの送信要求確認を行う
            if( pO0Ack = '0' ) then
              pO0Request <= '1';
              cRAddr <= (others => '0');
            else
              pO0Request <= '0';
              State <= SEND_DATA;
              cRAddr <= cRAddr + 1;
            end if;

          when SEND_DATA =>
            -- UPLパケットを送信する
            SendCount <= SendCount+1;
            if CONV_INTEGER(SendCount) = 0 then
              pO0Enable <= '1';
            elsif CONV_INTEGER(SendCount) = cLength then
              pO0Enable <= '0';
              State <= IDLE;
              pBusy <= '0';
            end if;
            pO0Data <= cRData;
            cRAddr <= cRAddr + 1;

          when others =>
            State <= IDLE;

        end case;
      end if;
    end if;
  end process;
end RTL;
