package net.wasamon.mailer.views;

import javax.mail.Folder;
import javax.mail.MessagingException;

import net.wasamon.mailer.composite.SummaryComposite;
import net.wasamon.mailer.driver.ClientData;
import net.wasamon.mailer.driver.IMAPClient;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.ViewPart;

/**
 * This sample class demonstrates how to plug-in a new workbench view. The view
 * shows data obtained from the model. The sample creates a dummy model on the
 * fly, but a real implementation would connect to the model available either in
 * this or another plug-in (e.g. the workspace). The view is connected to the
 * model using a content provider.
 * <p>
 * The view uses a label provider to define how model objects should be
 * presented in the view. Each view can present the same model objects using
 * different labels and icons, if needed. Alternatively, a single label provider
 * can be shared between views in order to ensure that objects of the same type
 * are presented in the same way everywhere.
 * <p>
 */

public class MailFolderView extends ViewPart {
	private TableViewer viewer;

	private Action actionFetchFolder;

	// private Action action2;
	private Action doubleClickAction;

	private ViewContentProvider provider;

	/*
	 * The content provider class is responsible for providing objects to the
	 * view. It can wrap existing objects in adapters or simply return objects
	 * as-is. These objects may be sensitive to the current input of the view,
	 * or ignore it and always show the same content (like Task List, for
	 * example).
	 */
	class ViewContentProvider implements IStructuredContentProvider {
		private String[] folders;

		public ViewContentProvider() {
			folders = new String[] {};
		}

		public void inputChanged(Viewer v, Object oldInput, Object newInput) {
		}

		public void dispose() {
		}

		public Object[] getElements(Object parent) {
			return folders;
		}

		public void setElement(String[] data) {
			folders = data;
		}
	}

	class ViewLabelProvider extends LabelProvider implements
			ITableLabelProvider {
		public String getColumnText(Object obj, int index) {
			return getText(obj);
		}

		public Image getColumnImage(Object obj, int index) {
			return getImage(obj);
		}

		public Image getImage(Object obj) {
			return PlatformUI.getWorkbench().getSharedImages().getImage(
					ISharedImages.IMG_OBJ_ELEMENT);
		}
	}

	class NameSorter extends ViewerSorter {
	}

	/**
	 * The constructor.
	 */
	public MailFolderView() {
	}

	/**
	 * This is a callback that will allow us to create the viewer and initialize
	 * it.
	 */
	public void createPartControl(Composite parent) {
		provider = new ViewContentProvider();
		viewer = new TableViewer(parent, SWT.MULTI | SWT.H_SCROLL
				| SWT.V_SCROLL);
		viewer.setContentProvider(provider);
		viewer.setLabelProvider(new ViewLabelProvider());
		viewer.setSorter(new NameSorter());
		viewer.setInput(getViewSite());
		makeActions();
		hookContextMenu();
		hookDoubleClickAction();
		contributeToActionBars();
	}

	private void hookContextMenu() {
		MenuManager menuMgr = new MenuManager("#PopupMenu");
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(IMenuManager manager) {
				MailFolderView.this.fillContextMenu(manager);
			}
		});
		Menu menu = menuMgr.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);
		getSite().registerContextMenu(menuMgr, viewer);
	}

	private void contributeToActionBars() {
		IActionBars bars = getViewSite().getActionBars();
		fillLocalPullDown(bars.getMenuManager());
		fillLocalToolBar(bars.getToolBarManager());
	}

	private void fillLocalPullDown(IMenuManager manager) {
		manager.add(actionFetchFolder);
		manager.add(new Separator());
		// manager.add(action2);
	}

	private void fillContextMenu(IMenuManager manager) {
		manager.add(actionFetchFolder);
		// manager.add(action2);
		// Other plug-ins can contribute there actions here
		manager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
	}

	private void fillLocalToolBar(IToolBarManager manager) {
		manager.add(actionFetchFolder);
		// manager.add(action2);
	}

	private IViewPart getViewPart(String id) {
		IWorkbenchPage workBenchPage = getSite().getWorkbenchWindow().getActivePage();
		try {
			return workBenchPage.showView(id);
		} catch (PartInitException e) {
			MessageDialog.openError(getSite().getShell(),
					"Mail Folder", "cannot find configration view: "
							+ e.getMessage());
		}
		return null;
	}

	private ClientData getClientData() {
		IViewPart part = getViewPart("net.wasamon.mailer.views.mailConfigView");
		if(part != null){
			return ((MailConfigView)part).getClientInfo().getIMAPClientData();
		}else{
			return null;
		}
	}
	
	private MailSummaryView getSummaryView(){
		IViewPart part = getViewPart("net.wasamon.mailer.views.mailSummaryView");
		if(part != null){
			return (MailSummaryView)part;
		}else{
			return null;
		}
	}

	private void makeActions() {
		actionFetchFolder = new Action() {
			public void run() {
				try {
					ClientData data = getClientData();
					if(data == null){
						return;
					}
					Folder[] folders = IMAPClient.getInstance().getFolderList(data);
					String[] d = new String[folders.length];
					for (int i = 0; i < folders.length; i++) {
						d[i] = folders[i].getFullName();
					}
					provider.setElement(d);
					viewer.refresh();
				} catch (MessagingException e) {
					MessageDialog.openError(getSite().getShell(), "Mailer",
							"cannot fetch directory: " + e.getMessage());
				}
			}
		};
		actionFetchFolder.setText("Fetch Folders");
		actionFetchFolder.setToolTipText("Fetch All Mail Folders");

		doubleClickAction = new Action() {
			public void run() {
				ISelection selection = viewer.getSelection();
				Object obj = ((IStructuredSelection) selection)
						.getFirstElement();
				setAndUpdateFolder(obj.toString());
			}
		};
	}

	private void setAndUpdateFolder(String folder) {
		SummaryComposite summary = getSummaryView().getSummaryComposite();
		summary.setAndUpdate(folder);
	}

	private void hookDoubleClickAction() {
		viewer.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {
				doubleClickAction.run();
			}
		});
	}

	/**
	 * Passing the focus request to the viewer's control.
	 */
	public void setFocus() {
		viewer.getControl().setFocus();
	}
}