package net.wasamon.mailer.driver;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Date;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.mail.Address;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Part;
import javax.mail.internet.AddressException;
import javax.mail.internet.ContentType;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeUtility;
import javax.mail.internet.ParseException;

public class MessageData {

	Address[] toAddr;
	Address fromAddr;
	Address[] ccAddr;
	Address[] bccAddr;
	String subject;
	String body;
	Date date;
	String xMailer;
	String userAgent;
	String plainMessage;
	boolean fMultiPart;
	String mesgID;
	String references;
	String inReplyTo;
	Message originalMesg;
	
	ArrayList<MultiPartData> mimeObject;
	ArrayList<File> attachFileList;

	public MessageData(Message mesg) throws MessagingException, IOException{
		fMultiPart = false;
		this.originalMesg = mesg;
		dumpPart(mesg);
	}
	
	public MessageData(String toAddr, String fromAddr, String subject) throws AddressException{
		this.toAddr = InternetAddress.parse(toAddr, false);
		Address[] a = InternetAddress.parse(fromAddr, false);
		if(a != null && a.length > 0){
			this.fromAddr = a[0];
		}
		this.subject = subject;
		ccAddr = null;
		bccAddr = null;
		body = "";
		fMultiPart = false;
	}
	
	public void setCcAddr(String addr) throws AddressException{
		ccAddr = InternetAddress.parse(addr, false);
	}
	public void setBccAddr(String addr) throws AddressException{
		bccAddr = InternetAddress.parse(addr, false);
	}
	public void setMessage(String mesg){
		this.body = mesg;
	}
	public void setReferences(String r){
		this.references = r;
	}
	public void setInReplyTo(String s){
		this.inReplyTo = s;
	}
	
	public String getPlainMessage(){
		return plainMessage;
	}
	public String getReference(){
		return references;
	}
	public String getMessageID(){
		return mesgID;
	}
	public String getMessage(){
		return body;
	}
	public boolean isMultiPart(){
		return fMultiPart;
	}
	public void setMultiPart(boolean flag){
		fMultiPart = flag;
	}
	public void setAttachFileList(ArrayList<File> list){
		attachFileList = list;
	}
	public ArrayList<File> getAttachFileList(){
		return attachFileList;
	}
	
	public String getToAddrStr(){
		return getAddressArrayAsStr(toAddr);
	}
	public String getCcAddrStr(){
		return getAddressArrayAsStr(ccAddr);
	}

	public static String getAddrStr(Address a){
		String str = a.toString();
		str = toDecodedString(str);
		return str;
	}
	
	public String getFromAddrStr(){
		return getAddrStr(fromAddr);
	}

	public Date getDate(){
		return date;
	}
	
	public String getSubject(){
		return subject;
	}

	private String getAddressArrayAsStr(Address[] a){
		String tmp = "";
		if(a == null) return tmp;
		for (int j = 0; j < a.length; j++) {
			tmp += "," + getAddrStr(a[j]);
		}
		if(tmp.length() > 1){
			tmp = tmp.substring(1);
		}
		return tmp;
	}
	
	private void dumpEnvelope(Message m) throws MessagingException{
		// FROM
		fromAddr = m.getFrom()[0];
		// TO
		toAddr = m.getRecipients(Message.RecipientType.TO);
		// CC
		ccAddr = m.getRecipients(Message.RecipientType.CC);
		// SUBJECT
		subject = toDecodedString(m.getSubject());
		// DATE
		date = m.getSentDate();
		// X-MAILER
		String[] hdrs = m.getHeader("X-Mailer");
		if (hdrs != null){
			xMailer = hdrs[0];
		}
		hdrs = m.getHeader("User-Agent");
		if(hdrs != null){
			userAgent = hdrs[0];
		}
		String[] msgid = m.getHeader("Message-ID");
		if(msgid != null){
			mesgID = msgid[0];
		}
		String[] refs = m.getHeader("References");
		if(refs != null){
			references = refs[0];
		}
	}

	private void pr(StringBuffer buf, String str){
		buf.append(str + "\n");
	}
	
	private void dumpPart(Part p) throws MessagingException, IOException {
		if (p instanceof Message){
			dumpEnvelope((Message) p);
		}
		StringBuffer buf = new StringBuffer();
		if (p.isMimeType("text/plain")) {
			fMultiPart = false;
			pr(buf, " ");
			plainMessage = (String)p.getContent();
			if(plainMessage != null){
				buf.append(plainMessage);
			}
		} else if (p.isMimeType("multipart/*")) {
			fMultiPart = true;
			Multipart mp = (Multipart) p.getContent();
			int count = mp.getCount();
			mimeObject = new ArrayList<MultiPartData>();
			for (int i = 0; i < count; i++){
				dumpPart(buf, mp.getBodyPart(i));
			}
		} else { // Null Data
			mimeObject = new ArrayList<MultiPartData>();
			String ct = p.getContentType();
			String type = "mime type";
			try {
				type = (new ContentType(ct)).toString();
			} catch (ParseException pex) {
				type = "unknown mime type";
			}
			Object o = p.getContent();
			if (o instanceof String) {
				mimeObject.add(new MultiPartData(type, o));
				pr(buf, "[ <" + type + ">]");
				buf.append((String) o);
				pr(buf, "");
			} else if (o instanceof InputStream) {
				String filename = "mime file";
				filename = p.getFileName();
				mimeObject.add(new MultiPartData(type, filename, o));
				pr(buf, "");
				pr(buf, "[ " + filename + " <" + type + ">]");
				pr(buf, "");
			} else {
				pr(buf, "This is an unknown type");
				pr(buf, "---------------------------");
				pr(buf, o.toString());
			}
		}
		body = buf.toString();
	}	
	
	public static Pattern encodedPattern = Pattern.compile("(^.*)(\\=\\?.*\\?\\=)(.*$)");
	public static String toDecodedString(String s){
		if(s == null){
			return "";
		}
		Matcher m = encodedPattern.matcher(s);
		String tmp = "";
		if(m.matches()){
			if(m.group(1) != null && m.group(1).length() > 0){
				tmp += toDecodedString(m.group(1));
			}
			if(m.group(2) != null && m.group(2).length() > 0){
				try {
					tmp += MimeUtility.decodeText(m.group(2));
				} catch (UnsupportedEncodingException e) {
					tmp += m.group(2);
				}
			}
			if(m.group(3) != null && m.group(3).length() > 0){
				tmp += toDecodedString(m.group(3));
			}
		}else{
			tmp = s;
		}
		return tmp;
	}
	
	private void dumpPart(StringBuffer buf, Part p) throws MessagingException, IOException {
		String ct = p.getContentType();
		String type = "mime type";
		try {
			type = (new ContentType(ct)).toString();
		} catch (ParseException pex) {
			type = "unknown mime type";
		}
		type = toDecodedString(type);
		if (p.isMimeType("text/plain")) {
			pr(buf, " ");
			pr(buf, "[ <" + type + ">]");
			plainMessage = (String)p.getContent();
			buf.append(plainMessage);
			pr(buf, "");
		} else if (p.isMimeType("multipart/*")) {
			fMultiPart = true;
			Multipart mp = (Multipart) p.getContent();
			int count = mp.getCount();
			for (int i = 0; i < count; i++){
				dumpPart(buf, mp.getBodyPart(i));
			}
		} else if (p.isMimeType("message/rfc822")) {
			pr(buf, "This is a Nested Message");
			pr(buf, "---------------------------");
			dumpPart(buf, (Part) p.getContent());
			pr(buf, "---------------------------");
		} else { // Null Data
			Object o = p.getContent();
			if (o instanceof String) {
				mimeObject.add(new MultiPartData(type, o));
				pr(buf, "[ <" + type + ">]");
				buf.append((String) o);
				pr(buf, "");
			} else if (o instanceof InputStream) {
				String filename = "mime_file"; 
				filename = toDecodedString(p.getFileName());
				mimeObject.add(new MultiPartData(type, filename, o));
				pr(buf, "");
				pr(buf, "[ " + filename + " <" + type + ">]");
				pr(buf, "");
			} else {
				pr(buf, "This is an unknown type");
				pr(buf, "---------------------------");
				pr(buf, o.toString());
			}
		}
		body = buf.toString();
	}

	public String getHeaderString(){
		StringBuffer sb = new StringBuffer();
		sb.append("To: " + getToAddrStr() + "\n");
		sb.append("Cc: " + getCcAddrStr() + "\n");
		sb.append("From: " + getFromAddrStr() + "\n");
		sb.append("Subject: " + getSubject() + "\n");
		sb.append("Date: " + (date != null ? date.toString() : "UNKNOWN") + "\n");
		if(xMailer != null){
			sb.append("X-Mailer: " + xMailer + "\n");
		}
		if(userAgent != null){
			sb.append("User-Agent: " + userAgent + "\n");
		}
		return sb.toString();
	}
	
	public int getMimeObjectSize(){
		if(mimeObject == null){
			return 0;
		}else{
			return mimeObject.size();
		}
	}

	public MultiPartData getMimeObject(int i){
		return mimeObject.get(i);
	}

	public class MultiPartData{
		String type = null;
		String name = null;
		Object obj = null;
		byte[] savedAsByteData;
		String savedAsString;
		boolean fBitStream = false;
		boolean fSaveData = false;
		public MultiPartData(String type, String name, Object obj){
			this.type = type;
			this.name = name;
			this.obj = obj;
		}
		public MultiPartData(String type, Object obj){
			this.type = type;
			this.obj = obj;
		}
		public String toString(){
			if(name != null){
				return "[ " + name + " <" + type + "> ]"; 
			}else{
				return "[ <" + type + "> ]"; 
			}
		}
		public Object getObject(){
			return obj;
		}
		public String getName(){
			return name;
		}
		
		private void saveToFile0(File f) throws IOException{
			if(fBitStream == false){
				FileWriter writer = new FileWriter(f);
				writer.write((String)obj);
				writer.flush();
				writer.close();
			}else{
				FileOutputStream out = new FileOutputStream(f);
				out.write(savedAsByteData);
				out.flush();
				out.close();
			}
			
		}
		
		public void saveToFile(File f) throws IOException{
			if(fSaveData == false){
				if(obj instanceof InputStream){
					fBitStream = true;
					ByteArrayOutputStream buf = new ByteArrayOutputStream();
					InputStream in = (InputStream)obj;
					int len;
					byte b[] = new byte[1024];
					while ((len = in.read(b)) != -1){
						buf.write(b, 0, len);
					}
					buf.flush();
					savedAsByteData = buf.toByteArray();
				}else{
					savedAsString = (String)obj.toString();
				}
				fSaveData = true;
			}
			saveToFile0(f);
		}
		
	}
	
}
