package net.wasamon.blokus.javarock.tools;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;

public class RomGenerator {

	private final String jname, vname;
	private final int width;
	private final int num;

	public RomGenerator(String jname, String vname, int width, int num){
		this.jname = jname;
		this.vname = vname;
		this.width = width;
		this.num = num;
	}

	private String genLibrariesImport(){
		String s = "";
		s += "library IEEE;\n";
	    s += "use IEEE.std_logic_1164.all;\n";
	    s += "use IEEE.std_logic_arith.all;\n";
	    s += "use IEEE.std_logic_unsigned.all;\n";
	    return s;
	}
	
	private String genEntity(){
		String s = "";
		s += String.format("entity %s is\n", vname);
		s += "  generic (\n";
		s += String.format("    DEPTH : integer := %d;\n", (int)Math.ceil((Math.log(num)/Math.log(2.0))));
	    s += String.format("    WIDTH : integer := %d );\n", width);
	    s += "\n";
	    s += "  port (\n";
	    s += "    clk   : in  std_logic;\n";
	    s += "    reset : in  std_logic;\n";
	    s += "    data_raddr : in  std_logic_vector(31 downto 0);\n";
	    s += String.format("    data_rdata : out std_logic_vector(%d downto 0) );\n", width-1);
	    s += "\n";
	    s += String.format("end %s;\n", vname);
	    s += "\n";
	    return s;
	}
	
	private String genArchitecture(String value){
		String s = "";
	    s += String.format("architecture RTL of %s is\n", vname);
	    s += "\n";
	    s += String.format("  type rom_type is array (0 to %d) of std_logic_vector (WIDTH-1 downto 0);\n", num-1);
	    s += "  constant ROM: rom_type := (\n";
	    for(String s0: value.split("\n")){
	    	s += "  " + s0 + "\n";
	    }
	    s += "  );";
	    s += String.format("  signal q : std_logic_vector(%d downto 0);", width-1);
	    s += "\n";
	    s += "begin  -- RTL\n";
	    s += "\n";
	    s += "  data_rdata <= q;\n";
	    s += "\n";
	    s += "  process (clk)\n";
	    s += "  begin\n";
	    s += "    if clk'event and clk = '1' then\n";
	    s += "      q <= ROM(conv_integer(data_raddr(DEPTH-1 downto 0)));\n";
	    s += "    end if;\n";
	    s += "  end process;\n";
	    s += "\n";
	    s += "end RTL;\n";
	    return s;
	}
	
	private String decode(String value){
		String s = "";
		String sep0 = "", sep1 = "";
		for(String s0: value.split("\n")){
			s += sep0;
			sep1 = "";
			for(String s1: s0.trim().split(",")){
				int v = Integer.parseInt(s1.trim().replaceAll("\"", ""), 2);
				switch(width){
				case 8:
					s += sep1 + String.format("(byte)0x%02x", v);
					break;
				case 16:
					s += sep1 + String.format("(short)0x%04x", v);
					break;
				default:
					s += sep1 + String.format("0x%08x", v);
					break;
				}
				sep1 = ", ";
			}
			sep0 = ",\n";
		}
		return s;
	}
	
	private String genJavaWrapper(String pkg, String value){
		String s = "";
		
		s += "package " + pkg + ";\n";
		s += "import net.wasamon.javarock.libraries.VHDLSimpleLibrary;\n";
		s += "import net.wasamon.javarock.model.JavaRockComponentIface;\n";
		s += "import net.wasamon.javarock.model.vhdl.VHDLPort;\n";
		s += "import net.wasamon.javarock.model.vhdl.VHDLGenericParameter;\n";
		s += "import net.wasamon.javarock.model.vhdl.type.VHDLTypeBuilder;\n";

		s += String.format("public class %s extends VHDLSimpleLibrary implements JavaRockComponentIface{\n", jname);

		switch(width){
		case 8:
			s += String.format("  public byte[] data = new byte[]{\n%s\n};\n", decode(value)); break;
		case 16:
			s += String.format("  public byte[] data = new short[]{\n%s\n};\n", decode(value)); break;
		default:
			s += String.format("  public byte[] data = new int[]{\n%s\n};\n", decode(value)); break;
		}

		s += String.format("  public %s(String... args){\n", jname);
		s += String.format("    super(\"%s\", args);\n", vname);

		s += "    ports.add(new VHDLPort(this, \"clk\",     VHDLTypeBuilder.getStdLogic(), VHDLPort.Dir.IN));\n";
		s += "    ports.add(new VHDLPort(this, \"reset\",   VHDLTypeBuilder.getStdLogic(), VHDLPort.Dir.IN));\n";

		s += "    ports.add(new VHDLPort(this, \"data_raddr\", VHDLTypeBuilder.getStdLogicVector(31, 0), VHDLPort.Dir.IN));\n";
		s += String.format("    ports.add(new VHDLPort(this, \"data_rdata\", VHDLTypeBuilder.getStdLogicVector(%d, 0), VHDLPort.Dir.OUT));\n", width-1);
		s += "  }\n";
		s += "}\n";
		return s;
	}

	public void genVHDL(String value){
		OutputStreamWriter out = null;
		try{
			out = new OutputStreamWriter(new FileOutputStream(vname + ".vhd"));
			out.write(genLibrariesImport());
			out.write(genEntity());
			out.write(genArchitecture(value));
			out.flush();
			out.close();
		}catch(IOException e){
			e.printStackTrace();
		}
	}
	
	public void genJava(String pkg, String value){
		OutputStreamWriter out = null;
		try{
			out = new OutputStreamWriter(new FileOutputStream(jname + ".java"));
			out.write(genJavaWrapper(pkg, value));
			out.flush();
			out.close();
		}catch(IOException e){
			e.printStackTrace();
		}
	}


}
